# backend/api/v1/auth.py
from fastapi import APIRouter, Depends, HTTPException, status
from fastapi.security import OAuth2PasswordRequestForm
from sqlalchemy.orm import Session
from typing import Any
from datetime import timedelta

from core.database import get_db
from core.security import auth_manager, security
from core.models import Owner, Admin
from core.exceptions import AuthenticationError
from shared.schemas import Token, LoginRequest, TokenRefreshRequest

router = APIRouter()


@router.post("/login", response_model=Token)
async def login(
    login_data: LoginRequest,
    db: Session = Depends(get_db)
) -> Any:
    """Login endpoint for owners and admins"""
    user = None
    user_type = None
    
    # Try owner login
    owner = await auth_manager.authenticate_owner(
        login_data.username, login_data.password, db
    )
    if owner:
        user = owner
        user_type = "owner"
    
    # Try admin login if not owner
    if not user:
        admin = await auth_manager.authenticate_admin(
            login_data.username, login_data.password, db
        )
        if admin:
            user = admin
            user_type = "admin"
    
    if not user:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Incorrect username or password",
            headers={"WWW-Authenticate": "Bearer"},
        )
    
    # Create tokens
    access_token = security.create_access_token(
        data={"sub": str(user.id), "user_type": user_type}
    )
    refresh_token = security.create_refresh_token(
        data={"sub": str(user.id), "user_type": user_type}
    )
    
    # Update last login
    if user_type == "owner":
        user.last_login = datetime.utcnow()
    else:
        user.last_login = datetime.utcnow()
    db.commit()
    
    return {
        "access_token": access_token,
        "refresh_token": refresh_token,
        "token_type": "bearer",
        "user_type": user_type,
        "user_id": user.id,
        "username": user.username if user_type == "owner" else user.admin_username
    }


@router.post("/refresh", response_model=Token)
async def refresh_token(
    refresh_data: TokenRefreshRequest,
    db: Session = Depends(get_db)
) -> Any:
    """Refresh access token using refresh token"""
    payload = security.decode_token(refresh_data.refresh_token)
    if not payload or payload.get("type") != "refresh":
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid refresh token"
        )
    
    user_id = int(payload.get("sub"))
    user_type = payload.get("user_type")
    
    # Verify user still exists
    if user_type == "owner":
        user = db.query(Owner).filter(Owner.id == user_id).first()
    else:
        user = db.query(Admin).filter(Admin.id == user_id).first()
    
    if not user:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="User not found"
        )
    
    # Create new access token
    access_token = security.create_access_token(
        data={"sub": str(user.id), "user_type": user_type}
    )
    
    return {
        "access_token": access_token,
        "refresh_token": refresh_data.refresh_token,
        "token_type": "bearer",
        "user_type": user_type,
        "user_id": user.id,
        "username": user.username if user_type == "owner" else user.admin_username
    }


@router.post("/logout")
async def logout(
    token: str = Depends(oauth2_scheme),
    db: Session = Depends(get_db)
) -> Any:
    """Logout endpoint - blacklist token"""
    # Implementation would blacklist the token
    return {"message": "Logged out successfully"}


@router.post("/verify-2fa")
async def verify_2fa(
    code: str,
    current_user = Depends(get_current_user)
) -> Any:
    """Verify 2FA code for owners"""
    if not isinstance(current_user, Owner):
        raise HTTPException(status_code=403, detail="Only owners can use 2FA")
    
    # Verify 2FA code (implementation would use pyotp)
    # valid = pyotp.TOTP(current_user.two_factor_secret).verify(code)
    
    return {"verified": True}